import math

# -------------------------
# Configuration Parameters
# -------------------------
SCALE = 10**10
SEED_VECTORS = 4             # minimal on-chain seed vectors
FOLD_DOUBLINGS = [2]*24      # 4 → 16,777,216 vectors
FIBS = [1,1,2,3,5,8,13,21,34,55,89,144,233,377,610]
PHI = 1.6180339887

# Seed params for integrity (hex string)
SEED_PARAMS = [
    ("0", "0x812"),
    ("1", "0x502"),
    ("2", "0x310"),
    ("3", "0x192")
]

# -------------------------
# Mini-Block Compression / On-Demand Expansion
# -------------------------
def reconstruct_vector(idx, user_id, seed_vectors=SEED_VECTORS):
    """
    Reconstruct a single vector on-demand.
    idx: vector index (0..16,777,215)
    user_id: deterministic per-user variation
    Returns a dict with r_dim and omega
    """
    # Compute folding level
    fold_level = int(math.log2(max(1, idx // seed_vectors)))
    
    # Fibonacci-based delta
    fib_delta = FIBS[fold_level % len(FIBS)]

    # Base r_dim progression (phi evolution)
    r_dim_base = (fold_level * PHI + user_id) % 256
    r_dim = r_dim_base / 255

    # Omega decay and index variation
    omega_base = 1e-8
    omega = omega_base * (0.9999 ** fold_level) + ((idx * 7 + user_id) % 1000 + fib_delta) * 1e-12

    # Optional per-vector seed key
    vector_key = f"user_{user_id}_vec_{idx}"

    return {
        "r_dim": r_dim,
        "omega": omega,
        "private_key": vector_key
    }

# -------------------------
# On-Demand Workspace Generator
# -------------------------
def user_workspace(user_id, start=0, end=1000):
    """
    Generate only a slice of user workspace on-demand.
    start/end: vector indices
    """
    return [reconstruct_vector(idx, user_id) for idx in range(start, end)]

# -------------------------
# Example Usage
# -------------------------
if __name__ == "__main__":
    # Example: reconstruct first 10 vectors for user 1
    workspace_slice = user_workspace(user_id=1, start=0, end=10)
    for idx, vec in enumerate(workspace_slice):
        print(f"Vector {idx}: {vec}")
